#
# Sim1Functions.R
#
# simulation functions for the first set of simulations
#
# 
#===============================================================================================#
#
# Sim1Rep 
#
# Arguments :
#       SimDatPars : A list of parameters to pass to SimDatHubSimple
#                    i.e., a list containing:
#                     n,pfh,pnh,mnh,pfi,pni,alphaHub,alphaInd,betaFHub,betaNFHub,vrb
#       MethodsList: A list of lists with each list having the following:
#                     method = {W, WGCNA, KMEANS, SPC}
#                     methodLabel = a user supplied vector of names for the methods employed
#                                   e.g., "K2" for Kmeans with 2 clusters
#                     methodPars = a list of parameters, if reqd (i.e., will be empty for SPC)
#
#                          methodPars for W will list: delta,gamma
#                                     for WGCNA will list: delta,ncluster
#                                     for KMEANS will list: ncluster
#
#                    We will let nmeth= # of methods applied to the data
#                    where, nmeth=length(MethodsList)
#       seed: a single value to set a seed, interpreted as an integer.
#
# Output :    
#       Umat: a p x nmeth logical matrix 
#             where Umat[i,j] = 1 if the ith feature was assigned to FH by the jth method
#             note: the columns of Umat will be labeled by methodLabel
#                   the rows of Umat will be labeled with the featureID
#                   
#         
#
#===============================================================================================#



Sim1Rep=function(SimDatPars, # A list of parameters to pass to SimDatHubSimple                   
                 MethodsList, #A list of lists with each list having entries given above
                 seed ){

  set.seed(seed)
  
  # identify the number of methods
  nmeth=length(MethodsList)
  
  # simulate the data
  all=do.call(SimDatHubOld,SimDatPars)  #use original simulation schemes.another option:SimDatHubSimple
  dat=list(x=all$x,y=all$y)
  
  # analyze the data and store the results
  Umat=matrix(F,dim(dat$x)[1],nmeth)
  colnames(Umat)=rep(" ",nmeth)
  rownames(Umat)=all$featureID
  
  for(h in 1:nmeth){
    tmp=MethodsList[[h]]
    colnames(Umat)[h]=tmp$methodLabel
    
    if(tmp$method=="W.e"){
      Umat[,h]=do.call(WK,c(list(data=dat),tmp$methodPars))
    }

    if(tmp$method=="W.bt"){
      Umat[,h]=do.call(WK.bt,c(list(data=dat),tmp$methodPars))
    }

    if(tmp$method=="W.W"){
      Umat[,h]=do.call(WK.W,c(list(data=dat),tmp$methodPars))
    }
    
    if(tmp$method=="WGCNA"){
      Umat[,h]=do.call(WGC,c(list(data=dat),tmp$methodPars))
    }    
    
    if(tmp$method=="KMEANS"){
      Umat[,h]=do.call(Km,c(list(data=dat),tmp$methodPars))
    } 
    
    if(tmp$method=="SPC"){
      Umat[,h]=spc(data=dat)
    } 
    
    
  }
  
  return(Umat)
  
}
  
